import React, { useMemo, useState } from 'react';
import { useFleet } from '../contexts/dataContext.tsx';
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow,
} from '@/components/ui/table';
import { Input } from '@/components/ui/input';
import {
    Select,
    SelectTrigger,
    SelectValue,
    SelectContent,
    SelectItem,
} from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';

type SortKey =
    | 'registrationNumber'
    | 'brand'
    | 'model'
    | 'currentMileage'
    | 'mileageLimit'
    | 'location'
    | 'serviceStatus'
    | 'contractUtilization';

type SortDir = 'asc' | 'desc';

type VehicleType = NonNullable<ReturnType<typeof useFleet>['vehicles']>[number];

export default function VehicleTable({
                                         onSelect,
                                         height = 420,
                                     }: {
    onSelect?: (vehicle: VehicleType) => void;
    height?: number;
}) {
    const { vehicles, loading, errors, refreshVehicles } = useFleet();
    const [query, setQuery] = useState('');
    const [statusFilter, setStatusFilter] = useState<'all' | 'ok' | 'warning' | 'critical'>('all');
    const [sortKey, setSortKey] = useState<SortKey>('registrationNumber');
    const [sortDir, setSortDir] = useState<SortDir>('asc');

    const data = vehicles ?? [];

    const filtered = useMemo(() => {
        const q = query.trim().toLowerCase();
        return data.filter((v) => {
            if (statusFilter !== 'all' && v.serviceStatus !== statusFilter) return false;
            if (!q) return true;
            const hay = [
                v.registrationNumber,
                v.brand,
                v.model,
                v.location,
                String(v.currentMileage),
                String(v.mileageLimit),
            ]
                .join(' ')
                .toLowerCase();
            return hay.includes(q);
        });
    }, [data, query, statusFilter]);

    const sorted = useMemo(() => {
        const arr = [...filtered];
        arr.sort((a, b) => {
            const va = getValue(a, sortKey);
            const vb = getValue(b, sortKey);
            const cmp =
                typeof va === 'number' && typeof vb === 'number'
                    ? va - vb
                    : String(va).localeCompare(String(vb), undefined, { numeric: true, sensitivity: 'base' });
            return sortDir === 'asc' ? cmp : -cmp;
        });
        return arr;
    }, [filtered, sortKey, sortDir]);

    const onSort = (key: SortKey) => {
        if (key === sortKey) setSortDir((d) => (d === 'asc' ? 'desc' : 'asc'));
        else {
            setSortKey(key);
            setSortDir('asc');
        }
    };

    return (
        <div className="flex flex-col gap-3">
            <div className="flex items-center gap-3">
                <Input
                    placeholder="Search trucks…"
                    value={query}
                    onChange={(e) => setQuery(e.target.value)}
                    className="max-w-sm"
                />
                <Select value={statusFilter} onValueChange={(v) => setStatusFilter(v as any)}>
                    <SelectTrigger className="w-40">
                        <SelectValue placeholder="Status" />
                    </SelectTrigger>
                    <SelectContent>
                        <SelectItem value="all">All statuses</SelectItem>
                        <SelectItem value="ok">OK</SelectItem>
                        <SelectItem value="warning">Warning</SelectItem>
                        <SelectItem value="critical">Critical</SelectItem>
                    </SelectContent>
                </Select>
                <Button variant="secondary" onClick={() => void refreshVehicles()} disabled={loading}>
                    {loading ? 'Refreshing…' : 'Refresh'}
                </Button>
            </div>

            {errors.vehicles && (
                <div className="text-sm text-red-600">Vehicles error: {errors.vehicles.message}</div>
            )}

            <ScrollArea className="rounded-md border" style={{ height }}>
                <Table>
                    <TableHeader className="sticky top-0 z-10 bg-background">
                        <TableRow>
                            <SortableHead label="Registration" sortKey="registrationNumber" sortState={{ sortKey, sortDir }} onSort={onSort} />
                            <SortableHead label="Brand" sortKey="brand" sortState={{ sortKey, sortDir }} onSort={onSort} />
                            <SortableHead label="Model" sortKey="model" sortState={{ sortKey, sortDir }} onSort={onSort} />
                            <SortableHead label="Mileage (km)" numeric sortKey="currentMileage" sortState={{ sortKey, sortDir }} onSort={onSort} />
                            <SortableHead label="Limit (km)" numeric sortKey="mileageLimit" sortState={{ sortKey, sortDir }} onSort={onSort} />
                            <SortableHead label="Location" sortKey="location" sortState={{ sortKey, sortDir }} onSort={onSort} />
                            <SortableHead label="Service" sortKey="serviceStatus" sortState={{ sortKey, sortDir }} onSort={onSort} />
                            <SortableHead label="Utilization" numeric sortKey="contractUtilization" sortState={{ sortKey, sortDir }} onSort={onSort} />
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {sorted.length === 0 ? (
                            <TableRow>
                                <TableCell colSpan={8} className="text-center text-muted-foreground">
                                    {loading ? 'Loading…' : 'No trucks to display'}
                                </TableCell>
                            </TableRow>
                        ) : (
                            sorted.map((v) => (
                                <TableRow
                                    key={v.id}
                                    className="cursor-pointer hover:bg-muted/50"
                                    role="button"
                                    tabIndex={0}
                                    onClick={() => onSelect?.(v)}
                                    onKeyDown={(e) => {
                                        if (e.key === 'Enter' || e.key === ' ') {
                                            e.preventDefault();
                                            onSelect?.(v);
                                        }
                                    }}
                                    title="Click for details"
                                >
                                    <TableCell className="font-medium">{v.registrationNumber}</TableCell>
                                    <TableCell>{v.brand}</TableCell>
                                    <TableCell>{v.model}</TableCell>
                                    <TableCell className="text-right tabular-nums">{fmtNum(v.currentMileage)}</TableCell>
                                    <TableCell className="text-right tabular-nums">{fmtNum(v.mileageLimit)}</TableCell>
                                    <TableCell>{v.location}</TableCell>
                                    <TableCell>
                                        <TooltipProvider delayDuration={200}>
                                            <Tooltip>
                                                <TooltipTrigger asChild>
                          <span>
                            <StatusBadge status={v.serviceStatus} />
                          </span>
                                                </TooltipTrigger>
                                                <TooltipContent>
                                                    <p>Service status: {v.serviceStatus}</p>
                                                </TooltipContent>
                                            </Tooltip>
                                        </TooltipProvider>
                                    </TableCell>
                                    <TableCell className="text-right">
                                        <div className="flex items-center gap-2 justify-end">
                                            <Progress
                                                value={clamp(v.contractUtilization, 0, 100)}
                                                className="w-28 h-2"
                                            />
                                            <span className="text-xs tabular-nums">
                        {clamp(v.contractUtilization, 0, 100).toFixed(0)}%
                      </span>
                                        </div>
                                    </TableCell>
                                </TableRow>
                            ))
                        )}
                    </TableBody>
                </Table>
            </ScrollArea>
            <p className="text-xs text-muted-foreground">Tip: Click a row to open the vehicle details.</p>
        </div>
    );
}

function StatusBadge({ status }: { status: 'ok' | 'warning' | 'critical' }) {
    const cfg =
        status === 'ok'
            ? { className: 'bg-emerald-100 text-emerald-800', text: 'OK' }
            : status === 'warning'
                ? { className: 'bg-amber-100 text-amber-800', text: 'Warning' }
                : { className: 'bg-red-100 text-red-800', text: 'Critical' };
    return (
        <Badge variant="secondary" className={`${cfg.className} font-semibold`}>
            {cfg.text}
        </Badge>
    );
}

function SortableHead({
                          label,
                          sortKey,
                          sortState,
                          onSort,
                          numeric,
                      }: {
    label: string;
    sortKey: SortKey;
    sortState: { sortKey: SortKey; sortDir: SortDir };
    onSort: (k: SortKey) => void;
    numeric?: boolean;
}) {
    const active = sortState.sortKey === sortKey;
    const caret = active ? (sortState.sortDir === 'asc' ? '▲' : '▼') : ' ';
    return (
        <TableHead
            onClick={() => onSort(sortKey)}
            className={`select-none whitespace-nowrap ${numeric ? 'text-right' : 'text-left'} ${active ? 'bg-muted/70' : ''} cursor-pointer`}
            title={`Sort by ${label}`}
        >
            <span>{label}</span>
            <span className="ml-1 text-muted-foreground text-xs align-middle">{caret}</span>
        </TableHead>
    );
}

function getValue(v: VehicleType, key: SortKey): string | number {
    switch (key) {
        case 'registrationNumber':
            return v.registrationNumber;
        case 'brand':
            return v.brand;
        case 'model':
            return v.model;
        case 'currentMileage':
            return v.currentMileage;
        case 'mileageLimit':
            return v.mileageLimit;
        case 'location':
            return v.location;
        case 'serviceStatus':
            return v.serviceStatus;
        case 'contractUtilization':
            return v.contractUtilization;
        default:
            return '';
    }
}

function clamp(n: number, min: number, max: number) {
    return Math.max(min, Math.min(max, n));
}

function fmtNum(n: number) {
    return new Intl.NumberFormat(undefined, { maximumFractionDigits: 0 }).format(n);
}