"use client";

import { useState } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { MapPinIcon, ArrowRightIcon } from "lucide-react";

interface RouteChange {
  id: string;
  truckRegistration: string;
  fromCity: string;
  toCity: string;
  date: string;
  status: "pending" | "approved" | "rejected";
}

export default function RouteChanges() {
  const [changes, setChanges] = useState<RouteChange[]>([
    {
      id: "1",
      truckRegistration: "WA 12345",
      fromCity: "Warszawa",
      toCity: "Kraków",
      date: "2025-11-01",
      status: "approved",
    },
    {
      id: "2",
      truckRegistration: "KR 67890",
      fromCity: "Gdańsk",
      toCity: "Wrocław",
      date: "2025-11-03",
      status: "pending",
    },
  ]);

  const [truckRegistration, setTruckRegistration] = useState("");
  const [fromCity, setFromCity] = useState("");
  const [toCity, setToCity] = useState("");
  const [date, setDate] = useState("");

  const polishCities = [
    "Warszawa",
    "Kraków",
    "Wrocław",
    "Poznań",
    "Gdańsk",
    "Szczecin",
    "Bydgoszcz",
    "Lublin",
    "Katowice",
    "Białystok",
  ];

  const handleAddChange = () => {
    if (truckRegistration && fromCity && toCity && date) {
      const newChange: RouteChange = {
        id: Date.now().toString(),
        truckRegistration,
        fromCity,
        toCity,
        date,
        status: "pending",
      };
      setChanges([...changes, newChange]);
      setTruckRegistration("");
      setFromCity("");
      setToCity("");
      setDate("");
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "approved":
        return "bg-green-500/10 text-green-500 border-green-500/20";
      case "pending":
        return "bg-yellow-500/10 text-yellow-500 border-yellow-500/20";
      case "rejected":
        return "bg-red-500/10 text-red-500 border-red-500/20";
      default:
        return "";
    }
  };

  return (
    <div className="grid gap-6 md:grid-cols-2">
      <Card>
        <CardHeader>
          <CardTitle>Request Route Change</CardTitle>
          <CardDescription>Submit a new route change request</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="truck-reg">Truck Registration</Label>
            <Input
              id="truck-reg"
              placeholder="e.g., WA 12345"
              value={truckRegistration}
              onChange={(e) => setTruckRegistration(e.target.value)}
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="from-city">From City</Label>
            <Select value={fromCity} onValueChange={setFromCity}>
              <SelectTrigger id="from-city">
                <SelectValue placeholder="Select origin city" />
              </SelectTrigger>
              <SelectContent>
                {polishCities.map((city) => (
                  <SelectItem key={city} value={city}>
                    {city}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <div className="space-y-2">
            <Label htmlFor="to-city">To City</Label>
            <Select value={toCity} onValueChange={setToCity}>
              <SelectTrigger id="to-city">
                <SelectValue placeholder="Select destination city" />
              </SelectTrigger>
              <SelectContent>
                {polishCities.map((city) => (
                  <SelectItem key={city} value={city}>
                    {city}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <div className="space-y-2">
            <Label htmlFor="change-date">Change Date</Label>
            <Input
              id="change-date"
              type="date"
              value={date}
              onChange={(e) => setDate(e.target.value)}
            />
          </div>
          <Button onClick={handleAddChange} className="w-full">
            <MapPinIcon className="mr-2 h-4 w-4" />
            Submit Route Change
          </Button>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle>Route Change History</CardTitle>
          <CardDescription>View all route change requests</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {changes.map((change) => (
              <div
                key={change.id}
                className="rounded-lg border border-border bg-muted/50 p-4"
              >
                <div className="flex items-center justify-between mb-2">
                  <p className="font-semibold text-foreground">
                    {change.truckRegistration}
                  </p>
                  <Badge
                    className={getStatusColor(change.status)}
                    variant="outline"
                  >
                    {change.status}
                  </Badge>
                </div>
                <div className="flex items-center gap-2 text-sm text-muted-foreground">
                  <span>{change.fromCity}</span>
                  <ArrowRightIcon className="h-4 w-4" />
                  <span>{change.toCity}</span>
                  <span className="ml-auto">{change.date}</span>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
