import React from 'react';
import type { Vehicle, Alert, Route } from '../contexts/dataContext.tsx';
import { Sheet, SheetContent, SheetHeader, SheetTitle, SheetDescription } from '@/components/ui/sheet';
import { Separator } from '@/components/ui/separator';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Progress } from '@/components/ui/progress';
import { ScrollArea } from '@/components/ui/scroll-area';

export default function VehicleSheet({
                                         open,
                                         vehicle,
                                         alerts,
                                         routes,
                                         onOpenChange,
                                     }: {
    open: boolean;
    vehicle: Vehicle | null;
    alerts: Alert[];
    routes: Route[];
    onOpenChange: (open: boolean) => void;
}) {
    return (
        <Sheet open={open} onOpenChange={onOpenChange}>
            <SheetContent side="left" className="w-[380px] sm:w-[420px] p-0">
                <SheetHeader className="p-4 pb-2 border-b">
                    <SheetTitle className="flex flex-col">
            <span className="text-base">
              {vehicle ? vehicle.registrationNumber : 'Vehicle'}
            </span>
                        {vehicle && (
                            <span className="text-xs text-muted-foreground">
                {vehicle.brand} {vehicle.model}
              </span>
                        )}
                    </SheetTitle>
                    <SheetDescription className="sr-only">Vehicle details</SheetDescription>
                </SheetHeader>

                <ScrollArea className="h-[calc(100vh-65px)]">
                    <div className="p-4 space-y-4">
                        {!vehicle ? (
                            <div className="text-sm text-muted-foreground">Select a vehicle to see details.</div>
                        ) : (
                            <>
                                <Card>
                                    <CardHeader className="pb-2">
                                        <CardTitle className="text-sm">Overview</CardTitle>
                                    </CardHeader>
                                    <CardContent className="space-y-3">
                                        <div className="text-sm">
                                            Location: <span className="font-semibold">{vehicle.location}</span>
                                        </div>
                                        <div className="text-sm">
                                            Mileage: <span className="font-semibold">{fmtNum(vehicle.currentMileage)}</span> km
                                            {' / '}
                                            Limit: <span className="font-semibold">{fmtNum(vehicle.mileageLimit)}</span> km
                                        </div>
                                        <div className="text-sm">
                                            Service status: <StatusBadge status={vehicle.serviceStatus} />
                                        </div>
                                        <div className="space-y-1">
                                            <div className="flex justify-between text-xs text-muted-foreground">
                                                <span>Contract utilization</span>
                                                <span className="tabular-nums">
                          {clamp(vehicle.contractUtilization, 0, 100).toFixed(0)}%
                        </span>
                                            </div>
                                            <Progress
                                                value={clamp(vehicle.contractUtilization, 0, 100)}
                                                className="h-2"
                                            />
                                        </div>
                                    </CardContent>
                                </Card>

                                <div className="space-y-2">
                                    <div className="text-sm font-medium">Related Alerts</div>
                                    <Separator />
                                    <div className="space-y-2">
                                        {alerts.length ? (
                                            alerts.map((a) => (
                                                <div key={a.id} className="flex items-start gap-3 py-1">
                                                    <SeverityBadge sev={a.severity} />
                                                    <div className="flex-1">
                                                        <div className="text-xs font-semibold">{a.type}</div>
                                                        <div className="text-sm text-muted-foreground">{a.message}</div>
                                                    </div>
                                                    <time className="text-xs text-muted-foreground">{a.timestamp}</time>
                                                </div>
                                            ))
                                        ) : (
                                            <div className="text-sm text-muted-foreground">No alerts for this vehicle</div>
                                        )}
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <div className="text-sm font-medium">Assigned Routes</div>
                                    <Separator />
                                    <div className="space-y-2">
                                        {routes.length ? (
                                            routes.slice(0, 12).map((r) => (
                                                <div key={r.id} className="py-1">
                                                    <div className="text-xs font-semibold">
                                                        {r.date}: {r.start} → {r.end}
                                                    </div>
                                                    <div className="text-sm text-muted-foreground">
                                                        {r.distance} km, {r.duration} min — status: {r.status}
                                                    </div>
                                                </div>
                                            ))
                                        ) : (
                                            <div className="text-sm text-muted-foreground">No routes for this vehicle</div>
                                        )}
                                    </div>
                                </div>
                            </>
                        )}
                    </div>
                </ScrollArea>
            </SheetContent>
        </Sheet>
    );
}

function StatusBadge({ status }: { status: Vehicle['serviceStatus'] }) {
    const cfg =
        status === 'ok'
            ? { className: 'bg-emerald-100 text-emerald-800', text: 'OK' }
            : status === 'warning'
                ? { className: 'bg-amber-100 text-amber-800', text: 'Warning' }
                : { className: 'bg-red-100 text-red-800', text: 'Critical' };
    return <Badge variant="secondary" className={`${cfg.className} font-semibold`}>{cfg.text}</Badge>;
}

function SeverityBadge({ sev }: { sev: 'low' | 'medium' | 'high' }) {
    const cfg =
        sev === 'low'
            ? { className: 'bg-indigo-100 text-indigo-800', text: 'Low' }
            : sev === 'medium'
                ? { className: 'bg-amber-100 text-amber-800', text: 'Medium' }
                : { className: 'bg-red-100 text-red-800', text: 'High' };
    return <Badge variant="secondary" className={`${cfg.className} font-semibold`}>{cfg.text}</Badge>;
}

function clamp(n: number, min: number, max: number) {
    return Math.max(min, Math.min(max, n));
}
function fmtNum(n: number) {
    return new Intl.NumberFormat(undefined, { maximumFractionDigits: 0 }).format(n);
}