"use client"

import { useState, useRef } from "react"
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Upload, File, X, CheckCircle, AlertCircle } from "lucide-react"
import { cn } from "@/lib/utils"

interface UploadedFile {
    original_filename: string
    saved_filename: string
    content_type: string
    size_bytes: number
    path: string
}

interface FileUploadProps {
    onFilesUploaded?: (files: UploadedFile[]) => void
    multiple?: boolean
    maxSizeMB?: number
    acceptedTypes?: string
}

export function FileUpload({
                               onFilesUploaded,
                               multiple = true,
                               maxSizeMB = 20,
                               acceptedTypes = "*/*"
                           }: FileUploadProps) {
    const [files, setFiles] = useState<File[]>([])
    const [uploadedFiles, setUploadedFiles] = useState<UploadedFile[]>([])
    const [isUploading, setIsUploading] = useState(false)
    const [dragActive, setDragActive] = useState(false)
    const fileInputRef = useRef<HTMLInputElement>(null)

    const handleFiles = (selectedFiles: FileList | null) => {
        if (!selectedFiles) return

        const fileArray = Array.from(selectedFiles)

        // Filter files by size if maxSizeMB is set
        const validFiles = fileArray.filter(file => {
            if (file.size > maxSizeMB * 1024 * 1024) {
                alert(`File ${file.name} exceeds maximum size of ${maxSizeMB}MB`)
                return false
            }
            return true
        })

        if (multiple) {
            setFiles(prev => [...prev, ...validFiles])
        } else {
            setFiles(validFiles.slice(0, 1))
        }
    }

    const handleDrag = (e: React.DragEvent) => {
        e.preventDefault()
        e.stopPropagation()

        if (e.type === "dragenter" || e.type === "dragover") {
            setDragActive(true)
        } else if (e.type === "dragleave") {
            setDragActive(false)
        }
    }

    const handleDrop = (e: React.DragEvent) => {
        e.preventDefault()
        e.stopPropagation()
        setDragActive(false)

        if (e.dataTransfer.files && e.dataTransfer.files.length > 0) {
            handleFiles(e.dataTransfer.files)
        }
    }

    const removeFile = (index: number) => {
        setFiles(prev => prev.filter((_, i) => i !== index))
    }

    const uploadFiles = async () => {
        if (files.length === 0) return

        setIsUploading(true)

        try {
            const formData = new FormData()
            files.forEach(file => {
                formData.append("files", file)
            })

            const response = await fetch("/api/files/upload", {
                method: "POST",
                body: formData,
            })

            if (!response.ok) {
                throw new Error(`Upload failed: ${response.statusText}`)
            }

            const result = await response.json()
            setUploadedFiles(result.uploaded)

            if (onFilesUploaded) {
                onFilesUploaded(result.uploaded)
            }

            // Clear files after successful upload
            setFiles([])

        } catch (error) {
            console.error("Upload error:", error)
            alert("Failed to upload files. Please try again.")
        } finally {
            setIsUploading(false)
        }
    }

    const formatFileSize = (bytes: number) => {
        if (bytes === 0) return "0 Bytes"
        const k = 1024
        const sizes = ["Bytes", "KB", "MB", "GB"]
        const i = Math.floor(Math.log(bytes) / Math.log(k))
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i]
    }

    const getFileIcon = (fileType: string) => {
        if (fileType.startsWith("image/")) return "🖼️"
        if (fileType.startsWith("video/")) return "🎥"
        if (fileType.startsWith("audio/")) return "🎵"
        if (fileType.includes("pdf")) return "📄"
        if (fileType.includes("word") || fileType.includes("document")) return "📝"
        if (fileType.includes("zip") || fileType.includes("archive")) return "📦"
        return "📁"
    }

    return (
        <Card>
            <CardHeader>
                <CardTitle className="flex items-center gap-2">
                    <Upload className="h-5 w-5 text-amber-600" />
                    File Upload
                </CardTitle>
                <CardDescription>
                    Upload files to the server. Maximum file size: {maxSizeMB}MB
                    {!multiple && " (Single file only)"}
                </CardDescription>
            </CardHeader>

            <CardContent className="space-y-4">
                {/* Upload Zone */}
                <div
                    className={cn(
                        "border-2 border-dashed rounded-lg p-6 text-center transition-colors cursor-pointer",
                        dragActive
                            ? "border-amber-400 bg-amber-50 dark:bg-amber-950/20"
                            : "border-amber-200 hover:border-amber-300 bg-amber-25 dark:bg-amber-950/10"
                    )}
                    onDragEnter={handleDrag}
                    onDragLeave={handleDrag}
                    onDragOver={handleDrag}
                    onDrop={handleDrop}
                    onClick={() => fileInputRef.current?.click()}
                >
                    <input
                        ref={fileInputRef}
                        type="file"
                        multiple={multiple}
                        accept={acceptedTypes}
                        onChange={(e) => handleFiles(e.target.files)}
                        className="hidden"
                    />

                    <Upload className="h-8 w-8 text-amber-400 mx-auto mb-2" />
                    <p className="text-sm text-amber-700 dark:text-amber-300 font-medium">
                        Drop files here or click to browse
                    </p>
                    <p className="text-xs text-muted-foreground mt-1">
                        Supports {multiple ? "multiple files" : "single file"} up to {maxSizeMB}MB each
                    </p>
                </div>

                {/* Selected Files List */}
                {files.length > 0 && (
                    <div className="space-y-2">
                        <h4 className="text-sm font-medium text-amber-700 dark:text-amber-300">
                            Selected Files ({files.length})
                        </h4>
                        <div className="space-y-2 max-h-40 overflow-y-auto">
                            {files.map((file, index) => (
                                <div
                                    key={index}
                                    className="flex items-center justify-between p-3 border border-amber-200 rounded-lg bg-amber-50 dark:bg-amber-950/20 dark:border-amber-700"
                                >
                                    <div className="flex items-center gap-3 flex-1 min-w-0">
                                        <span className="text-lg">{getFileIcon(file.type)}</span>
                                        <div className="flex-1 min-w-0">
                                            <p className="text-sm font-medium truncate">{file.name}</p>
                                            <p className="text-xs text-muted-foreground">
                                                {formatFileSize(file.size)}
                                            </p>
                                        </div>
                                    </div>
                                    <Button
                                        variant="ghost"
                                        size="icon"
                                        onClick={() => removeFile(index)}
                                        className="h-8 w-8 text-amber-600 hover:text-amber-800 hover:bg-amber-100"
                                    >
                                        <X className="h-4 w-4" />
                                    </Button>
                                </div>
                            ))}
                        </div>

                        {/* Upload Button */}
                        <Button
                            onClick={uploadFiles}
                            disabled={isUploading || files.length === 0}
                            className="w-full bg-amber-500 hover:bg-amber-600 text-white"
                        >
                            {isUploading ? (
                                <>
                                    <div className="h-4 w-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2" />
                                    Uploading...
                                </>
                            ) : (
                                <>
                                    <Upload className="h-4 w-4 mr-2" />
                                    Upload {files.length} File{files.length !== 1 ? "s" : ""}
                                </>
                            )}
                        </Button>
                    </div>
                )}

                {/* Uploaded Files */}
                {uploadedFiles.length > 0 && (
                    <div className="space-y-2">
                        <h4 className="text-sm font-medium text-green-700 dark:text-green-300 flex items-center gap-2">
                            <CheckCircle className="h-4 w-4" />
                            Successfully Uploaded ({uploadedFiles.length})
                        </h4>
                        <div className="space-y-2 max-h-40 overflow-y-auto">
                            {uploadedFiles.map((file, index) => (
                                <div
                                    key={index}
                                    className="flex items-center justify-between p-3 border border-green-200 rounded-lg bg-green-50 dark:bg-green-950/20 dark:border-green-700"
                                >
                                    <div className="flex items-center gap-3 flex-1 min-w-0">
                                        <File className="h-4 w-4 text-green-600" />
                                        <div className="flex-1 min-w-0">
                                            <p className="text-sm font-medium truncate">{file.original_filename}</p>
                                            <div className="flex items-center gap-2 mt-1">
                                                <Badge variant="outline" className="text-xs">
                                                    {formatFileSize(file.size_bytes)}
                                                </Badge>
                                                <span className="text-xs text-muted-foreground">
                          {file.saved_filename}
                        </span>
                                            </div>
                                        </div>
                                    </div>
                                    <CheckCircle className="h-4 w-4 text-green-600 flex-shrink-0" />
                                </div>
                            ))}
                        </div>
                    </div>
                )}
            </CardContent>
        </Card>
    )
}
