import React, { useEffect } from 'react';
import type { Vehicle, Alert, Route } from '../contexts/dataContext.tsx';

interface VehicleTrayProps {
    open: boolean;
    vehicle: Vehicle | null;
    alerts: Alert[];
    routes: Route[];
    onClose: () => void;
}

export default function VehicleTray({ open, vehicle, alerts, routes, onClose }: VehicleTrayProps) {
    // Close on Escape
    useEffect(() => {
        const onKey = (e: KeyboardEvent) => {
            if (e.key === 'Escape') onClose();
        };
        if (open) window.addEventListener('keydown', onKey);
        return () => window.removeEventListener('keydown', onKey);
    }, [open, onClose]);

    return (
        <>
            {/* Backdrop */}
            <div
                onClick={onClose}
                style={{
                    position: 'fixed',
                    inset: 0,
                    background: open ? 'rgba(0,0,0,0.32)' : 'transparent',
                    opacity: open ? 1 : 0,
                    pointerEvents: open ? 'auto' : 'none',
                    transition: 'opacity 200ms ease',
                    zIndex: 999,
                }}
            />

            {/* Tray */}
            <aside
                role="dialog"
                aria-hidden={!open}
                style={{
                    position: 'fixed',
                    inset: '0 auto 0 0',
                    width: 380,
                    maxWidth: '90vw',
                    background: '#fff',
                    boxShadow: '2px 0 24px rgba(0,0,0,0.15)',
                    transform: open ? 'translateX(0)' : 'translateX(-102%)',
                    transition: 'transform 220ms ease',
                    zIndex: 1000,
                    display: 'flex',
                    flexDirection: 'column',
                }}
            >
                <header
                    style={{
                        padding: '14px 16px',
                        borderBottom: '1px solid #eee',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'space-between',
                        gap: 8,
                    }}
                >
                    <div style={{ display: 'flex', flexDirection: 'column' }}>
                        <strong style={{ fontSize: 16 }}>
                            {vehicle ? vehicle.registrationNumber : 'Vehicle'}
                        </strong>
                        {vehicle && (
                            <span style={{ fontSize: 12, color: '#666' }}>
                {vehicle.brand} {vehicle.model}
              </span>
                        )}
                    </div>
                    <button
                        onClick={onClose}
                        aria-label="Close"
                        style={{
                            border: '1px solid #ddd',
                            background: '#fafafa',
                            borderRadius: 6,
                            padding: '6px 10px',
                            cursor: 'pointer',
                        }}
                    >
                        Close
                    </button>
                </header>

                <div style={{ padding: 16, overflow: 'auto' }}>
                    {!vehicle ? (
                        <div style={{ color: '#666' }}>Select a vehicle to see details.</div>
                    ) : (
                        <>
                            <section style={{ marginBottom: 16 }}>
                                <h3 style={h3}>Overview</h3>
                                <ul style={ul}>
                                    <li>
                                        Location: <b>{vehicle.location}</b>
                                    </li>
                                    <li>
                                        Mileage: <b>{fmtNum(vehicle.currentMileage)}</b> km / Limit:{' '}
                                        <b>{fmtNum(vehicle.mileageLimit)}</b> km
                                    </li>
                                    <li style={{ display: 'flex', alignItems: 'center', gap: 8 }}>
                                        Contract utilization:
                                        <div style={utilWrap}>
                                            <div style={utilBarBg}>
                                                <div
                                                    style={{
                                                        ...utilBarFill,
                                                        width: `${Math.min(100, Math.max(0, vehicle.contractUtilization))}%`,
                                                    }}
                                                />
                                            </div>
                                            <span style={{ fontVariantNumeric: 'tabular-nums' }}>
                        {vehicle.contractUtilization.toFixed(0)}%
                      </span>
                                        </div>
                                    </li>
                                    <li>
                                        Service status: <StatusBadge status={vehicle.serviceStatus} />
                                    </li>
                                </ul>
                            </section>

                            <section style={{ marginBottom: 16 }}>
                                <h3 style={h3}>Related Alerts</h3>
                                {alerts.length ? (
                                    <ul style={listReset}>
                                        {alerts.map((a) => (
                                            <li key={a.id} style={row}>
                        <span style={{ minWidth: 72 }}>
                          <SeverityBadge sev={a.severity} />
                        </span>
                                                <div style={{ flex: 1 }}>
                                                    <div style={{ fontWeight: 600, fontSize: 13 }}>{a.type}</div>
                                                    <div style={{ color: '#555', fontSize: 13 }}>{a.message}</div>
                                                </div>
                                                <time style={{ color: '#777', fontSize: 12 }}>{a.timestamp}</time>
                                            </li>
                                        ))}
                                    </ul>
                                ) : (
                                    <i style={{ color: '#666' }}>No alerts for this vehicle</i>
                                )}
                            </section>

                            <section>
                                <h3 style={h3}>Assigned Routes</h3>
                                {routes.length ? (
                                    <ul style={listReset}>
                                        {routes.slice(0, 10).map((r) => (
                                            <li key={r.id} style={row}>
                                                <div style={{ flex: 1 }}>
                                                    <div style={{ fontWeight: 600, fontSize: 13 }}>
                                                        {r.date}: {r.start} → {r.end}
                                                    </div>
                                                    <div style={{ color: '#555', fontSize: 13 }}>
                                                        {r.distance} km, {r.duration} min — status: {r.status}
                                                    </div>
                                                </div>
                                            </li>
                                        ))}
                                    </ul>
                                ) : (
                                    <i style={{ color: '#666' }}>No routes for this vehicle</i>
                                )}
                            </section>
                        </>
                    )}
                </div>
            </aside>
        </>
    );
}

function StatusBadge({ status }: { status: Vehicle['serviceStatus'] }) {
    const cfg =
        status === 'ok'
            ? { bg: '#e7f8ec', fg: '#0a7a2a', text: 'OK' }
            : status === 'warning'
                ? { bg: '#fff6e6', fg: '#a56500', text: 'Warning' }
                : { bg: '#fde8e8', fg: '#a50013', text: 'Critical' };
    return (
        <span
            style={{
                background: cfg.bg,
                color: cfg.fg,
                padding: '2px 8px',
                borderRadius: 999,
                fontSize: 12,
                fontWeight: 600,
            }}
        >
      {cfg.text}
    </span>
    );
}

function SeverityBadge({ sev }: { sev: 'low' | 'medium' | 'high' }) {
    const cfg =
        sev === 'low'
            ? { bg: '#eef2ff', fg: '#3730a3', text: 'Low' }
            : sev === 'medium'
                ? { bg: '#fff6e6', fg: '#a56500', text: 'Medium' }
                : { bg: '#fde8e8', fg: '#a50013', text: 'High' };
    return (
        <span
            style={{
                background: cfg.bg,
                color: cfg.fg,
                padding: '2px 8px',
                borderRadius: 999,
                fontSize: 12,
                fontWeight: 600,
            }}
        >
      {cfg.text}
    </span>
    );
}

const h3: React.CSSProperties = { margin: '0 0 8px', fontSize: 14 };

const ul: React.CSSProperties = {
    margin: '0 0 0 16px',
    padding: 0,
    display: 'grid',
    gap: 6,
    fontSize: 14,
};

const listReset: React.CSSProperties = { margin: 0, padding: 0, listStyle: 'none', display: 'grid', gap: 8 };

const row: React.CSSProperties = {
    display: 'flex',
    alignItems: 'flex-start',
    gap: 8,
    padding: '8px 0',
    borderBottom: '1px solid #f0f0f0',
};

const utilWrap: React.CSSProperties = { display: 'flex', alignItems: 'center', gap: 8 };
const utilBarBg: React.CSSProperties = { height: 8, width: 120, background: '#eee', borderRadius: 999, overflow: 'hidden' };
const utilBarFill: React.CSSProperties = { height: '100%', background: '#3b82f6' };

function fmtNum(n: number) {
    return new Intl.NumberFormat(undefined, { maximumFractionDigits: 0 }).format(n);
}