import React, { useMemo } from 'react';
import { type Change, type Route, type Kpis } from '../contexts/dataContext.tsx';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Separator } from '@/components/ui/separator';
import { Badge } from '@/components/ui/badge';

export default function SwapsCard({
                                      changes,
                                      routes,
                                      kpis,
                                      height = 360,
                                  }: {
    changes: Change[] | null;
    routes: Route[] | null;
    kpis: Kpis | null;
    height?: number;
}) {
    const swaps = useMemo(() => {
        const list = (changes ?? []).filter((c) => {
            const desc = c.description.toLowerCase();
            return c.type === 'assignment' || desc.includes('swap') || desc.includes('zamian');
        });
        // newest first
        return list.sort((a, b) => (a.timestamp < b.timestamp ? 1 : -1));
    }, [changes]);

    const routeById = useMemo(() => {
        const map = new Map<string, Route>();
        (routes ?? []).forEach((r) => map.set(r.id, r));
        return map;
    }, [routes]);

    return (
        <Card className="flex flex-col">
            <CardHeader className="pb-2">
                <CardTitle className="text-lg">Swaps</CardTitle>
                <CardDescription>
                    Total swaps: {kpis?.totalSwaps ?? '-'}
                </CardDescription>
            </CardHeader>
            <Separator />
            <CardContent className="p-0">
                <ScrollArea className={`h-[${height}px]`}>
                    <div className="p-4 pt-3 space-y-3">
                        {swaps.length ? (
                            swaps.map((s) => {
                                const r = s.routeId ? routeById.get(s.routeId) : undefined;
                                return (
                                    <div key={s.id} className="rounded-md border p-3">
                                        <div className="flex items-start justify-between gap-3">
                                            <div className="space-y-1">
                                                <div className="flex items-center gap-2">
                                                    <Badge variant="secondary" className="bg-sky-100 text-sky-800">
                                                        Swap
                                                    </Badge>
                                                    <span className="text-xs uppercase tracking-wide text-muted-foreground">
                            {s.type}
                          </span>
                                                </div>
                                                <div className="text-sm">{s.description}</div>
                                                <div className="text-xs text-muted-foreground">
                                                    By {s.user}
                                                    {r ? (
                                                        <>
                                                            {' • '}
                                                            <span className="font-medium">
                                {r.date}: {r.start} → {r.end}
                              </span>
                                                        </>
                                                    ) : null}
                                                </div>
                                            </div>
                                            <time className="text-xs text-muted-foreground whitespace-nowrap">
                                                {s.timestamp}
                                            </time>
                                        </div>
                                    </div>
                                );
                            })
                        ) : (
                            <div className="text-sm text-muted-foreground">No swaps</div>
                        )}
                    </div>
                </ScrollArea>
            </CardContent>
        </Card>
    );
}