import React, { useMemo, useState } from 'react';
import { useFleet, type Vehicle, type Alert, type Route } from '../contexts/dataContext.tsx';
import VehicleTable from './truck-registration.tsx';
import VehicleSheet from './VehicleSheet';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { ScrollArea } from '@/components/ui/scroll-area';
import SwapsCard from './SwapsCard';
import {FileUpload} from "@/components/UploadButton.tsx";
import axios from "axios";

export function Dashboard() {
    const { routes, alerts, changes, kpis, loading, errors, lastFetchedAt, refreshAll } = useFleet();

    const [selectedVehicle, setSelectedVehicle] = useState<Vehicle | null>(null);
    const [sheetOpen, setSheetOpen] = useState(false);

    const vehicleAlerts: Alert[] = useMemo(() => {
        if (!selectedVehicle) return [];
        return (alerts ?? []).filter((a) => a.vehicleId === selectedVehicle.id);
    }, [alerts, selectedVehicle]);

    const vehicleRoutes: Route[] = useMemo(() => {
        if (!selectedVehicle) return [];
        // In case assignedVehicle is id or registrationNumber
        return (routes ?? []).filter(
            (r) =>
                r.assignedVehicle === selectedVehicle.id ||
                r.assignedVehicle === selectedVehicle.registrationNumber
        );
    }, [routes, selectedVehicle]);

    return (
        <div className="p-4 space-y-6">
            <div className="flex items-center justify-between gap-3">
                <h1 className="text-2xl font-bold">Fleet Dashboard</h1>
                <div className="flex items-center gap-3">
                    <button onClick={() => {axios.post("/planner/run")}}>Rozpocznij obliczanie</button>
                    <button
                        className="inline-flex items-center justify-center rounded-md border px-3 py-2 text-sm"
                        onClick={() => void refreshAll()}
                        disabled={loading}
                    >
                        {loading ? 'Refreshing…' : 'Refresh All'}
                    </button>
                    <small className="text-muted-foreground">
                        Last fetched: {lastFetchedAt?.toLocaleString() ?? '—'}
                    </small>
                    <FileUpload/>
                </div>
            </div>

            <div className="space-y-2 text-red-600">
                {errors.routes && <div>Routes error: {errors.routes.message}</div>}
                {errors.alerts && <div>Alerts error: {errors.alerts.message}</div>}
                {errors.kpis && <div>KPIs error: {errors.kpis.message}</div>}
            </div>

            <section className="space-y-3">
                <h2 className="text-lg font-semibold">KPIs</h2>
                {kpis ? (
                    <div className="grid gap-3 sm:grid-cols-2 lg:grid-cols-4">
                        <Card>
                            <CardHeader className="pb-2">
                                <CardDescription>Vehicles in limit</CardDescription>
                                <CardTitle className="text-2xl">
                                    {kpis.vehiclesInLimit}/{kpis.totalVehicles}
                                </CardTitle>
                            </CardHeader>
                        </Card>
                        <Card>
                            <CardHeader className="pb-2">
                                <CardDescription>Total swaps</CardDescription>
                                <CardTitle className="text-2xl">{kpis.totalSwaps}</CardTitle>
                            </CardHeader>
                        </Card>
                        <Card>
                            <CardHeader className="pb-2">
                                <CardDescription>Avg time to limit</CardDescription>
                                <CardTitle className="text-2xl">{kpis.avgTimeToLimit}d</CardTitle>
                            </CardHeader>
                        </Card>
                        <Card>
                            <CardHeader className="pb-2">
                                <CardDescription>Contract utilization</CardDescription>
                                <CardTitle className="text-2xl">{kpis.contractUtilization}%</CardTitle>
                            </CardHeader>
                        </Card>
                    </div>
                ) : (
                    <i className="text-muted-foreground">No KPI data</i>
                )}
            </section>

            <section className="space-y-2">
                <h2 className="text-lg font-semibold">Trucks</h2>
                <VehicleTable
                    onSelect={(v) => {
                        setSelectedVehicle(v);
                        setSheetOpen(true);
                    }}
                />
            </section>

            {/* Routes, Alerts, Swaps side-by-side with equal heights */}
            <section className="grid grid-cols-1 gap-4 lg:grid-cols-2 xl:grid-cols-3">
                <Card className="flex flex-col">
                    <CardHeader className="pb-2">
                        <CardTitle className="text-lg">Routes</CardTitle>
                        <CardDescription>Planned, in progress and completed routes</CardDescription>
                    </CardHeader>
                    <Separator />
                    <CardContent className="p-0">
                        <ScrollArea className="h-[360px]">
                            <div className="p-4 pt-3 space-y-3">
                                {routes?.length ? (
                                    routes.map((r) => (
                                        <div key={r.id} className="rounded-md border p-3">
                                            <div className="flex items-start justify-between gap-3">
                                                <div className="space-y-1">
                                                    <div className="text-sm font-semibold">
                                                        {r.date}: {r.start} → {r.end}
                                                    </div>
                                                    <div className="text-sm text-muted-foreground">
                                                        {r.distance} km • {r.duration} min
                                                    </div>
                                                </div>
                                                <RouteStatusBadge status={r.status} />
                                            </div>
                                        </div>
                                    ))
                                ) : (
                                    <div className="text-sm text-muted-foreground">No routes</div>
                                )}
                            </div>
                        </ScrollArea>
                    </CardContent>
                </Card>

                <Card className="flex flex-col">
                    <CardHeader className="pb-2">
                        <CardTitle className="text-lg">Alerts</CardTitle>
                        <CardDescription>Fleet alerts by severity</CardDescription>
                    </CardHeader>
                    <Separator />
                    <CardContent className="p-0">
                        <ScrollArea className="h-[360px]">
                            <div className="p-4 pt-3 space-y-3">
                                {alerts?.length ? (
                                    alerts.map((a) => (
                                        <div key={a.id} className="rounded-md border p-3">
                                            <div className="flex items-start justify-between gap-3">
                                                <div className="space-y-1">
                                                    <div className="flex items-center gap-2">
                                                        <SeverityBadge sev={a.severity} />
                                                        <span className="text-xs uppercase tracking-wide text-muted-foreground">
                              {a.type}
                            </span>
                                                    </div>
                                                    <div className="text-sm">{a.message}</div>
                                                </div>
                                                <time className="text-xs text-muted-foreground whitespace-nowrap">
                                                    {a.timestamp}
                                                </time>
                                            </div>
                                        </div>
                                    ))
                                ) : (
                                    <div className="text-sm text-muted-foreground">No alerts</div>
                                )}
                            </div>
                        </ScrollArea>
                    </CardContent>
                </Card>

                <SwapsCard changes={changes ?? null} routes={routes ?? null} kpis={kpis ?? null} />
            </section>

            <VehicleSheet
                open={sheetOpen}
                vehicle={selectedVehicle}
                alerts={vehicleAlerts}
                routes={vehicleRoutes}
                onOpenChange={setSheetOpen}
            />
        </div>
    );
}

function RouteStatusBadge({ status }: { status: Route['status'] }) {
    const cfg =
        status === 'planned'
            ? { text: 'Planned', cls: 'bg-slate-100 text-slate-800' }
            : status === 'in_progress'
                ? { text: 'In progress', cls: 'bg-amber-100 text-amber-800' }
                : { text: 'Done', cls: 'bg-emerald-100 text-emerald-800' };
    return <Badge variant="secondary" className={`${cfg.cls}`}>{cfg.text}</Badge>;
}

function SeverityBadge({ sev }: { sev: 'low' | 'medium' | 'high' }) {
    const cfg =
        sev === 'low'
            ? { text: 'Low', cls: 'bg-indigo-100 text-indigo-800' }
            : sev === 'medium'
                ? { text: 'Medium', cls: 'bg-amber-100 text-amber-800' }
                : { text: 'High', cls: 'bg-red-100 text-red-800' };
    return <Badge variant="secondary" className={`${cfg.cls}`}>{cfg.text}</Badge>;
}