import React, {
    type PropsWithChildren,
    createContext,
    useCallback,
    useContext,
    useEffect,
    useMemo,
    useRef,
    useState,
} from 'react';

/* =========================
   Types (from your API spec)
   ========================= */

export type ServiceStatus = 'ok' | 'warning' | 'critical';

export interface Vehicle {
    id: string;
    registrationNumber: string;
    brand: string;
    model: string;
    currentMileage: number;
    location: string;
    serviceStatus: ServiceStatus;
    mileageLimit: number;
    contractUtilization: number; // e.g., 82.5
}

export interface Coordinates {
    lat: number;
    lng: number;
}

// Backend currently returns: planned | in-progress | completed
export type RawRouteStatus = 'planned' | 'in-progress' | 'completed';

// Frontend-normalized: planned | in_progress | done
export type RouteStatus = 'planned' | 'in_progress' | 'done';

export interface RawRoute {
    id: string;
    start: string;
    end: string;
    distance: number; // km
    duration: number; // minutes
    assignedVehicle: string | null;
    date: string; // YYYY-MM-DD
    status: RawRouteStatus;
    startCoordinates: Coordinates;
    endCoordinates: Coordinates;
}

export interface Route {
    id: string;
    start: string;
    end: string;
    distance: number;
    duration: number;
    assignedVehicle: string | null;
    date: string;
    status: RouteStatus; // normalized
    startCoordinates: Coordinates;
    endCoordinates: Coordinates;
}

export type AlertType = 'mileage' | 'service' | 'conflict';
export type AlertSeverity = 'low' | 'medium' | 'high';

export interface Alert {
    id: string;
    type: AlertType;
    severity: AlertSeverity;
    message: string;
    vehicleId: string | null;
    routeId: string | null;
    timestamp: string; // YYYY-MM-DD HH:MM
}

export type ChangeType = 'assignment' | 'update' | 'status';

export interface Change {
    id: string;
    timestamp: string; // YYYY-MM-DD HH:MM
    type: ChangeType;
    description: string;
    routeId: string | null;
    user: string;
}

export interface Kpis {
    vehiclesInLimit: number;
    totalVehicles: number;
    totalSwaps: number;
    avgTimeToLimit: number; // days
    contractUtilization: number; // percent (integer)
}

/* =========================
   Context shape
   ========================= */

type DataErrorMap = Partial<{
    vehicles: Error;
    routes: Error;
    alerts: Error;
    changes: Error;
    kpis: Error;
}>;

export interface FleetContextValue {
    // Data
    vehicles: Vehicle[] | null;
    routes: Route[] | null;
    alerts: Alert[] | null;
    changes: Change[] | null;
    kpis: Kpis | null;

    // State
    loading: boolean;
    errors: DataErrorMap;
    lastFetchedAt: Date | null;

    // Actions
    refreshAll: () => Promise<void>;
    refreshVehicles: () => Promise<void>;
    refreshRoutes: () => Promise<void>;
    refreshAlerts: () => Promise<void>;
    refreshChanges: () => Promise<void>;
    refreshKpis: () => Promise<void>;
}

/* =========================
   Context + Provider
   ========================= */

const FleetContext = createContext<FleetContextValue | undefined>(undefined);

export interface FleetProviderProps {
    apiBaseUrl?: string;
    autoLoad?: boolean; // default true
    pollIntervalMs?: number; // optional polling, default off
}

export function FleetProvider({
                                  apiBaseUrl,
                                  autoLoad = true,
                                  pollIntervalMs,
                                  children,
                              }: PropsWithChildren<FleetProviderProps>) {
    // Data
    const [vehicles, setVehicles] = useState<Vehicle[] | null>(null);
    const [routes, setRoutes] = useState<Route[] | null>(null);
    const [alerts, setAlerts] = useState<Alert[] | null>(null);
    const [changes, setChanges] = useState<Change[] | null>(null);
    const [kpis, setKpis] = useState<Kpis | null>(null);

    // State
    const [loading, setLoading] = useState<boolean>(false);
    const [errors, setErrors] = useState<DataErrorMap>({});
    const [lastFetchedAt, setLastFetchedAt] = useState<Date | null>(null);

    // Internals
    const baseUrlRef = useRef<string>(deriveInitialBase(apiBaseUrl));
    const abortRef = useRef<AbortController | null>(null);
    const pollRef = useRef<number | null>(null);

    // Keep base URL updated when prop changes
    useEffect(() => {
        baseUrlRef.current = deriveInitialBase(apiBaseUrl);
    }, [apiBaseUrl]);

    // Cleanup on unmount
    useEffect(() => {
        return () => {
            if (abortRef.current) abortRef.current.abort();
            if (pollRef.current !== null) {
                clearInterval(pollRef.current);
                pollRef.current = null;
            }
        };
    }, []);

    // API client bound to current base URL
    const client = useMemo(() => createClient(() => baseUrlRef.current), []);

    const refreshVehicles = useCallback(async () => {
        const ctrl = new AbortController();
        abortRef.current = ctrl;
        try {
            const data = await client.getVehicles(ctrl.signal);
            setVehicles(data);
            setErrors((e) => ({ ...e, vehicles: undefined }));
        } catch (err) {
            setErrors((e) => ({ ...e, vehicles: toError(err) }));
        }
    }, [client]);

    const refreshRoutes = useCallback(async () => {
        const ctrl = new AbortController();
        abortRef.current = ctrl;
        try {
            const data = await client.getRoutes(ctrl.signal);
            setRoutes(data);
            setErrors((e) => ({ ...e, routes: undefined }));
        } catch (err) {
            setErrors((e) => ({ ...e, routes: toError(err) }));
        }
    }, [client]);

    const refreshAlerts = useCallback(async () => {
        const ctrl = new AbortController();
        abortRef.current = ctrl;
        try {
            const data = await client.getAlerts(ctrl.signal);
            setAlerts(data);
            setErrors((e) => ({ ...e, alerts: undefined }));
        } catch (err) {
            setErrors((e) => ({ ...e, alerts: toError(err) }));
        }
    }, [client]);

    const refreshChanges = useCallback(async () => {
        const ctrl = new AbortController();
        abortRef.current = ctrl;
        try {
            const data = await client.getChanges(ctrl.signal);
            setChanges(data);
            setErrors((e) => ({ ...e, changes: undefined }));
        } catch (err) {
            setErrors((e) => ({ ...e, changes: toError(err) }));
        }
    }, [client]);

    const refreshKpis = useCallback(async () => {
        const ctrl = new AbortController();
        abortRef.current = ctrl;
        try {
            const data = await client.getKpis(ctrl.signal);
            setKpis(data);
            setErrors((e) => ({ ...e, kpis: undefined }));
        } catch (err) {
            setErrors((e) => ({ ...e, kpis: toError(err) }));
        }
    }, [client]);

    const refreshAll = useCallback(async () => {
        if (abortRef.current) abortRef.current.abort();
        const ctrl = new AbortController();
        abortRef.current = ctrl;

        setLoading(true);
        try {
            const [v, r, a, c, k] = await Promise.allSettled([
                client.getVehicles(ctrl.signal),
                client.getRoutes(ctrl.signal),
                client.getAlerts(ctrl.signal),
                client.getChanges(ctrl.signal),
                client.getKpis(ctrl.signal),
            ]);

            setErrors({});

            if (v.status === 'fulfilled') setVehicles(v.value);
            else setErrors((e) => ({ ...e, vehicles: toError(v.reason) }));

            if (r.status === 'fulfilled') setRoutes(r.value);
            else setErrors((e) => ({ ...e, routes: toError(r.reason) }));

            if (a.status === 'fulfilled') setAlerts(a.value);
            else setErrors((e) => ({ ...e, alerts: toError(a.reason) }));

            if (c.status === 'fulfilled') setChanges(c.value);
            else setErrors((e) => ({ ...e, changes: toError(c.reason) }));

            if (k.status === 'fulfilled') setKpis(k.value);
            else setErrors((e) => ({ ...e, kpis: toError(k.reason) }));

            setLastFetchedAt(new Date());
        } finally {
            setLoading(false);
        }
    }, [client]);

    // Initial load
    useEffect(() => {
        if (autoLoad) {
            void refreshAll();
        }
    }, [autoLoad, refreshAll]);

    // Optional polling
    useEffect(() => {
        if (!pollIntervalMs || pollIntervalMs <= 0) {
            if (pollRef.current !== null) {
                clearInterval(pollRef.current);
                pollRef.current = null;
            }
            return;
        }
        if (pollRef.current !== null) {
            clearInterval(pollRef.current);
            pollRef.current = null;
        }
        pollRef.current = window.setInterval(() => {
            void refreshAll();
        }, pollIntervalMs);
        return () => {
            if (pollRef.current !== null) {
                clearInterval(pollRef.current);
                pollRef.current = null;
            }
        };
    }, [pollIntervalMs, refreshAll]);

    const value = useMemo<FleetContextValue>(
        () => ({
            vehicles,
            routes,
            alerts,
            changes,
            kpis,
            loading,
            errors,
            lastFetchedAt,
            refreshAll,
            refreshVehicles,
            refreshRoutes,
            refreshAlerts,
            refreshChanges,
            refreshKpis,
        }),
        [
            vehicles,
            routes,
            alerts,
            changes,
            kpis,
            loading,
            errors,
            lastFetchedAt,
            refreshAll,
            refreshVehicles,
            refreshRoutes,
            refreshAlerts,
            refreshChanges,
            refreshKpis,
        ]
    );

    return <FleetContext.Provider value={value}>{children}</FleetContext.Provider>;
}

export function useFleet(): FleetContextValue {
    const ctx = useContext(FleetContext);
    if (!ctx) throw new Error('useFleet must be used within <FleetProvider>');
    return ctx;
}

/* =========================
   Helpers and API client
   ========================= */

function toError(err: unknown): Error {
    return err instanceof Error ? err : new Error(String(err));
}

function deriveInitialBase(explicit?: string): string {
    // Prefer prop, then Vite, then CRA, then fallback
    const base =
        explicit ||
        (typeof import.meta !== 'undefined' &&
            (import.meta as any).env?.VITE_API_BASE_URL) ||
        (typeof process !== 'undefined' &&
            (process as any).env?.REACT_APP_API_BASE_URL) ||
        'http://127.0.0.1:8000/api';
    return normalizeApiBase(base);
}

function normalizeApiBase(url: string): string {
    let u = (url || '').trim().replace(/\/+$/, '');
    if (!u.endsWith('/api')) u = `${u}/api`;
    return u;
}

function mapRouteStatus(raw: RawRouteStatus): RouteStatus {
    switch (raw) {
        case 'planned':
            return 'planned';
        case 'in-progress':
            return 'in_progress';
        case 'completed':
            return 'done';
        default:
            return 'planned';
    }
}

function mapRoute(raw: RawRoute): Route {
    return { ...raw, status: mapRouteStatus(raw.status) };
}

function createClient(getBaseUrl: () => string) {
    async function request<T>(path: string, init?: RequestInit, signal?: AbortSignal): Promise<T> {
        const res = await fetch(`${getBaseUrl()}${path}`, {
            ...init,
            signal,
            headers: {
                Accept: 'application/json',
                'Content-Type': 'application/json',
                ...(init?.headers || {}),
            },
        });
        if (!res.ok) {
            const msg = await safeText(res);
            throw new Error(`HTTP ${res.status} ${res.statusText} for ${path}${msg ? `: ${msg}` : ''}`);
        }
        if (res.status === 204) {
            return null as unknown as T;
        }
        return (await res.json()) as T;
    }

    async function safeText(res: Response) {
        try {
            return await res.text();
        } catch {
            return '';
        }
    }

    return {
        async getVehicles(signal?: AbortSignal): Promise<Vehicle[]> {
            return request<Vehicle[]>('/vehicles', undefined, signal);
        },
        async getRoutes(signal?: AbortSignal): Promise<Route[]> {
            const raw = await request<RawRoute[]>('/routes', undefined, signal);
            return raw.map(mapRoute);
        },
        async getAlerts(signal?: AbortSignal): Promise<Alert[]> {
            return request<Alert[]>('/alerts', undefined, signal);
        },
        async getChanges(signal?: AbortSignal): Promise<Change[]> {
            return request<Change[]>('/changes', undefined, signal);
        },
        async getKpis(signal?: AbortSignal): Promise<Kpis> {
            return request<Kpis>('/kpis', undefined, signal);
        },
    };
}