"use client";

import { useState } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { CalendarIcon, WrenchIcon } from "lucide-react";

interface ServiceRecord {
  id: string;
  truckRegistration: string;
  serviceType: string;
  date: string;
  status: "scheduled" | "completed" | "overdue";
}

export default function TruckService() {
  const [services, setServices] = useState<ServiceRecord[]>([
    {
      id: "1",
      truckRegistration: "WA 12345",
      serviceType: "Oil Change",
      date: "2025-11-05",
      status: "scheduled",
    },
    {
      id: "2",
      truckRegistration: "KR 67890",
      serviceType: "Brake Inspection",
      date: "2025-10-28",
      status: "overdue",
    },
    {
      id: "3",
      truckRegistration: "WA 12345",
      serviceType: "Tire Rotation",
      date: "2025-10-15",
      status: "completed",
    },
  ]);

  const [truckRegistration, setTruckRegistration] = useState("");
  const [serviceType, setServiceType] = useState("");
  const [date, setDate] = useState("");

  const handleAddService = () => {
    if (truckRegistration && serviceType && date) {
      const newService: ServiceRecord = {
        id: Date.now().toString(),
        truckRegistration,
        serviceType,
        date,
        status: "scheduled",
      };
      setServices([...services, newService]);
      setTruckRegistration("");
      setServiceType("");
      setDate("");
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "completed":
        return "bg-green-500/10 text-green-500 border-green-500/20";
      case "scheduled":
        return "bg-blue-500/10 text-blue-500 border-blue-500/20";
      case "overdue":
        return "bg-red-500/10 text-red-500 border-red-500/20";
      default:
        return "";
    }
  };

  return (
    <div className="grid gap-6 md:grid-cols-2">
      <Card>
        <CardHeader>
          <CardTitle>Schedule Service</CardTitle>
          <CardDescription>Add a new service appointment</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="truck">Truck Registration</Label>
            <Input
              id="truck"
              placeholder="e.g., WA 12345"
              value={truckRegistration}
              onChange={(e) => setTruckRegistration(e.target.value)}
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="service-type">Service Type</Label>
            <Select value={serviceType} onValueChange={setServiceType}>
              <SelectTrigger id="service-type">
                <SelectValue placeholder="Select service type" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="Oil Change">Oil Change</SelectItem>
                <SelectItem value="Brake Inspection">
                  Brake Inspection
                </SelectItem>
                <SelectItem value="Tire Rotation">Tire Rotation</SelectItem>
                <SelectItem value="Engine Diagnostic">
                  Engine Diagnostic
                </SelectItem>
                <SelectItem value="Full Service">Full Service</SelectItem>
              </SelectContent>
            </Select>
          </div>
          <div className="space-y-2">
            <Label htmlFor="date">Service Date</Label>
            <Input
              id="date"
              type="date"
              value={date}
              onChange={(e) => setDate(e.target.value)}
            />
          </div>
          <Button onClick={handleAddService} className="w-full">
            <CalendarIcon className="mr-2 h-4 w-4" />
            Schedule Service
          </Button>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle>Service Records</CardTitle>
          <CardDescription>
            Track maintenance and service history
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {services.map((service) => (
              <div
                key={service.id}
                className="flex items-center justify-between rounded-lg border border-border bg-muted/50 p-4"
              >
                <div className="flex items-center gap-3">
                  <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-primary/10">
                    <WrenchIcon className="h-5 w-5 text-primary" />
                  </div>
                  <div>
                    <p className="font-semibold text-foreground">
                      {service.truckRegistration}
                    </p>
                    <p className="text-sm text-muted-foreground">
                      {service.serviceType} • {service.date}
                    </p>
                  </div>
                </div>
                <Badge
                  className={getStatusColor(service.status)}
                  variant="outline"
                >
                  {service.status}
                </Badge>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
